from django.db import models
from django.contrib.auth.models import User
from django.utils import timezone
from accounts.models import BASE_URL, make_thumbnail, compress
from io import BytesIO
from PIL import Image
from django.core.files import File
from .company_scrapper import main
import uuid

def gen_uuid():
    return uuid.uuid4().hex

class Company(models.Model):
    name = models.CharField(max_length=200)
    description = models.TextField(blank=True, null=True)
    website = models.URLField(blank=True, null=True)
    image = models.ImageField(upload_to ='company_image_uploads/', blank=True, null=True)
    thumbnail = models.ImageField(upload_to ='company_image_uploads/', blank=True, null=True)
    image_url = models.CharField(max_length=200, blank=True, null=True)
    meta_data = models.JSONField(blank=True, null=True)
    datetime_created = models.DateTimeField(default=timezone.now)
    datetime_updated = models.DateTimeField(auto_now=True)

    class Meta:
         verbose_name_plural = 'Company'
         ordering = ('name',)
    
    def __str__(self):
        return f"<COMPANY : [{self.name}] />"

    def save(self, *args, **kwargs):
        new_image = compress(self.image)
        self.image = new_image
        super().save(*args, **kwargs)
    
    def get_absolute_url(self):
        return f'/company/{self.pk}/'
    
    def get_image_url(self):
        if self.image_url:
            return BASE_URL + '/' + self.image_url
        elif self.website:
            img, meta = main(self.website, f"{self.name.replace(' ', '_')}_{gen_uuid()}.png")
            self.image_url = img
            self.meta_data = meta
            self.save()
            return BASE_URL + '/' + self.image_url
        else:
            return ''
    
    
    def get_image(self):
        if self.image:
            return BASE_URL + self.image.url
        else:
            return ''
    

    def get_thumbnail(self):
        if self.thumbnail:
            return BASE_URL + self.thumbnail.url
        elif self.image:
            self.thumbnail = make_thumbnail(self.image)
            self.save()
            return BASE_URL + self.thumbnail.url
        else:
            return f'/static/assets/gif/link.gif'


STATUS_CHOICES = [
        ('Wishlist', 'Wishlist'),
        ('Applied', 'Applied'),
        ('Interview', 'Interview'),
        ('Offer', 'Offer'),
        ('Rejected', 'Rejected'),
    ]

class Job(models.Model):
    title = models.CharField(max_length=200)
    description = models.TextField()
    company = models.ForeignKey(Company, on_delete=models.CASCADE)
    link = models.URLField(blank=True, null=True)
    status = models.CharField(max_length=15, choices=STATUS_CHOICES, default='Applied')
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name='jobs')
    datetime_created = models.DateTimeField(default=timezone.now)
    datetime_updated = models.DateTimeField(auto_now=True)

    class Meta:
         verbose_name_plural = 'Job'
         ordering = ('id',)
         
    def __str__(self):
        return f"<JOB [{self.title}] />"
    
    def get_absolute_url(self):
        return f'/job/{self.pk}/'
    


class JobUpdates(models.Model):
    job = models.ForeignKey(Job, on_delete=models.CASCADE, related_name='update')
    from_action = models.CharField(max_length=15, choices=STATUS_CHOICES, default='Applied')
    to_action = models.CharField(max_length=15, choices=STATUS_CHOICES, default='Applied')
    update = models.TextField()
    datetime_created = models.DateTimeField(default=timezone.now)
    datetime_updated = models.DateTimeField(auto_now=True)

    class Meta:
         verbose_name_plural = 'Job Update'
         ordering = ('job', 'id')
         
    def __str__(self):
        return f"<JOB ACTION [{self.job} | from = {self.from_action} | to = {self.to_action} />"
    
    def get_absolute_url(self):
        return f'/job/{self.pk}/'
    


from django.core.validators import FileExtensionValidator
from django.core.exceptions import ValidationError
FILE_CHOICES = [
        ('Resume', 'Resume'),
        ('Cover Letter', 'Cover Letter'),
        ('Others', 'Others'),
    ]

def validate_image(fieldfile_obj):
    filesize = fieldfile_obj.file.size
    megabyte_limit = 1.0
    if filesize > megabyte_limit*1024*1024:
        raise ValidationError("Max file size is %sMB" % str(megabyte_limit))

""" JOB FILES """
class FilesModel(models.Model):       
    job = models.ForeignKey(Job, related_name='files', on_delete=models.CASCADE)
    file = models.FileField(validators=[FileExtensionValidator(['png','jpg','jpeg','pdf','doc','docx','cdr', 'xls', 'xlsx']), validate_image], upload_to='job_files_folder', blank=True, null=True)
    file_type = models.CharField(max_length=30, choices=FILE_CHOICES)
    description = models.TextField(default='')
    datetime_created = models.DateTimeField(default=timezone.now)
    datetime_updated = models.DateTimeField(auto_now=True)

    class Meta:
         verbose_name_plural = 'Files'
         ordering = ('job',)

    def __str__(self):
         return f"<JOB FILES [{self.job} | {self.file_type} />"
    

from django import forms
from typing import Any
class FilesForm(forms.ModelForm):    
    file = forms.FileField(widget=forms.FileInput(attrs={"class": "col-12 form-control mb-3"}))
    class Meta:
        model = FilesModel
        fields = ['file']

        def __init__(self, *args: Any, **kwargs: Any) -> None:
            super(FilesForm, self).__init__(*args, **kwargs)

            self.fields['file'].help_text = "File Max Size is 1M"

        def clean_image(self):
            file = self.cleaned_data.get('file', False)
            if file:
                if file._size > 4*1024*1024:
                    raise ValidationError("File file too large ( > 4mb )")
                return file
            else:
                raise ValidationError("Couldn't read uploaded file")
    
